﻿//-------------------------------------------------------------------------------------
// SID Monitor - Utility for Sudden Ionospheric Disturbances Monitoring Stations
// Copyright (C) 2005 - Lionel Loudet
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//-------------------------------------------------------------------------------------


using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Net;

namespace SID_monitor
{
    public partial class PanelDatabase : UserControl
    {
        public PanelDatabase()
        {
            InitializeComponent();

            // set the default folder to executable location
            this.openFileDialog.InitialDirectory = Application.StartupPath;

            this.textBoxRRDToolPath.Text = SID_monitor.Properties.Settings.Default.RRDToolPath;
            this.textBoxHistoryDatabaseFolder.Text = Path.GetFullPath(SID_monitor.Properties.Settings.Default.HistoryDatabaseFolder);
            this.textBoxRRDToolChannelsDatabase.Text = Path.GetFullPath(SID_monitor.Properties.Settings.Default.RRDToolDatabase);
            this.textBoxRRDToolGOESDatabase.Text = Path.GetFullPath(SID_monitor.Properties.Settings.Default.RRDToolGOES);

        }

        #region Properties
        public string RRDToolPath
        {
            get
            {
                return this.textBoxRRDToolPath.Text;
            }
        }


        public string RRDToolChannelsDatabase
        {
            get
            {
                return this.textBoxRRDToolChannelsDatabase.Text;
            }
        }

        public string RRDToolGOESDatabase
        {
            get
            {
                return this.textBoxRRDToolGOESDatabase.Text;
            }
        }

        public string HistoryDatabaseFolder
        {
            get
            {
                return this.textBoxHistoryDatabaseFolder.Text;
            }
        }
        #endregion

        #region Select File Methods
        private void buttonSelectRRDTool_Click(object sender, EventArgs e)
        {
            this.openFileDialog.DefaultExt = "exe";
            this.openFileDialog.Filter = "Executable Files|*.exe|All files|*.*";
            this.openFileDialog.Title = "Select RRDTool location";

            if (File.Exists(this.textBoxRRDToolPath.Text))
            {
                this.openFileDialog.FileName = this.textBoxRRDToolPath.Text;
            }

            if (this.openFileDialog.ShowDialog() == DialogResult.OK)
            {
                this.textBoxRRDToolPath.Text = this.openFileDialog.FileName;
            }

        }

        private void buttonSelectRRDToolChannelsDatabase_Click(object sender, EventArgs e)
        {
            this.openFileDialog.DefaultExt = "rrd";
            this.openFileDialog.Filter = "RRDTool Database|*.rrd|All files|*.*";
            this.openFileDialog.Title = "Select RRDTool Channels Database";
            if (File.Exists(this.textBoxRRDToolChannelsDatabase.Text))
            {
                this.openFileDialog.FileName = this.textBoxRRDToolChannelsDatabase.Text;
            }

            if (this.openFileDialog.ShowDialog() == DialogResult.OK)
            {
                this.textBoxRRDToolChannelsDatabase.Text = this.openFileDialog.FileName;
            }

        }

        private void buttonSelectRRDToolGOESDatabase_Click(object sender, EventArgs e)
        {
            this.openFileDialog.DefaultExt = "rrd";
            this.openFileDialog.Filter = "RRDTool Database|*.rrd|All files|*.*";
            this.openFileDialog.Title = "Select RRDTool GOES Database";
            if (File.Exists(this.textBoxRRDToolGOESDatabase.Text))
            {
                this.openFileDialog.FileName = this.textBoxRRDToolGOESDatabase.Text;
            }

            if (this.openFileDialog.ShowDialog() == DialogResult.OK)
            {
                this.textBoxRRDToolGOESDatabase.Text = this.openFileDialog.FileName;
            }

        }

        private void buttonHistoryDatabaseFolder_Click(object sender, EventArgs e)
        {
            //this.folderBrowserDialog.Description = 
            if (File.Exists(this.textBoxHistoryDatabaseFolder.Text))
            {
                this.folderBrowserDialog.SelectedPath = this.textBoxHistoryDatabaseFolder.Text;
            }

            if (this.folderBrowserDialog.ShowDialog() == DialogResult.OK)
            {
                this.textBoxHistoryDatabaseFolder.Text = this.folderBrowserDialog.SelectedPath;
            }

        }

        #endregion

        #region Values Validation Method
        private void textBoxFile_TextChanged(object sender, EventArgs e)
        {
            TextBox currentTextBox = (TextBox)sender;
            try
            {
                if (String.IsNullOrEmpty(Path.GetFileNameWithoutExtension(currentTextBox.Text)) ||
                         String.IsNullOrEmpty(Path.GetExtension(currentTextBox.Text)) ||
                         Path.GetFileName(currentTextBox.Text).IndexOfAny(Path.GetInvalidFileNameChars()) >= 0)
                {
                    warningProvider.SetError(currentTextBox, "");
                    errorProvider.SetError(currentTextBox, "Please enter a valid filename");
                    OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, false));
                }
                else if (!File.Exists(Path.GetFullPath(currentTextBox.Text)))
                {
                    warningProvider.SetError(currentTextBox, ""); 
                    errorProvider.SetError(currentTextBox, "This file does not exists");
                    OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, false));
                }
                else
                {
                    warningProvider.SetError(currentTextBox, "");
                    errorProvider.SetError(currentTextBox, "");
                    OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, true));
                }
            }
            catch (Exception ex)
            {
                errorProvider.SetError(currentTextBox, ex.ToString().Split('\n')[0]);
                OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, false));
            }

        }

        private void textBoxRRDToolDatabase_TextChanged(object sender, EventArgs e)
        {
            TextBox currentTextBox = (TextBox)sender;
            try
            {
                if (String.IsNullOrEmpty(Path.GetFileNameWithoutExtension(currentTextBox.Text)) ||
                     String.IsNullOrEmpty(Path.GetExtension(currentTextBox.Text)) ||
                     Path.GetFileName(currentTextBox.Text).IndexOfAny(Path.GetInvalidFileNameChars()) >= 0)
                {
                    warningProvider.SetError(currentTextBox, "");
                    errorProvider.SetError(currentTextBox, "Please enter a valid filename");
                    OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, false));
                }
                else if (!Directory.Exists(Path.GetDirectoryName(currentTextBox.Text)))
                {
                    warningProvider.SetError(currentTextBox, "");
                    errorProvider.SetError(currentTextBox, "This folder does not exists. Cannot create this database file.");
                    OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, false));
                }
                else
                {
                    errorProvider.SetError(currentTextBox, "");
                    OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, true));

                    if (!File.Exists(Path.GetFullPath(currentTextBox.Text)))
                    {
                        warningProvider.SetError(currentTextBox, "This file does not exists. It will be automatically created.");
                    }
                    else
                    {
                        warningProvider.SetError(currentTextBox, "");
                    }
                }
            }
            catch (Exception ex)
            {
                errorProvider.SetError(currentTextBox, ex.ToString().Split('\n')[0]);
                OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, false));
            }
        }

        private void textBoxHistoryFolder_TextChanged(object sender, EventArgs e)
        {
            TextBox currentTextBox = (TextBox)sender;
            if (!Directory.Exists(currentTextBox.Text))
            {
                errorProvider.SetError(currentTextBox, "This folder does not exists");
                OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, false));
            }
            else
            {
                errorProvider.SetError(currentTextBox, "");
                OnValidData(new validDataEventArgs("NodeDatabase", currentTextBox.Handle, true));
            }
        }
        #endregion

        #region ValidData Event
        public event ValidDataEventHandler ValidData;

        protected virtual void OnValidData(validDataEventArgs e)
        {
            if (ValidData != null)
            {
                // Invokes the delegates. 
                ValidData(this, e);
            }
        }
        #endregion






    }
}
